<?php
namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Models\SuratPermohonan;
use App\Models\PermohonanProjek;
use App\Models\Progress;
use App\Models\Aduan;
use App\Models\User;

use Carbon\Carbon;


class DashboardController extends Controller
{
    public function index()
    {
        // Get the current database connection from the session
        // $connectionName = session('db_conn', 'mysql'); // Default to 'mysql' if not set
        $totalSurat = SuratPermohonan::count();
        $surat = SuratPermohonan::where('status', 1)->count();
        $totalProjek = PermohonanProjek::count();
        $projek = PermohonanProjek::where('status', 1)->count();
        $totalAduan = Aduan::count();
        $aduan = Aduan::where('status', 1)->count();

        // Calculate approved count (assuming status 2 is approved)
        $approvedCount = SuratPermohonan::where('status', 2)->count() + 
                        PermohonanProjek::where('status', 2)->count();
        
        // Calculate response time analytics
        $responseTimeData = $this->calculateResponseTimeData();
        // dd($responseTimeData);

        // Get recent applications with response time
        $recentApplications = $this->getRecentApplicationsWithResponseTime();
        
        // Get trend data for the last 30 days
        $trendData = $this->getResponseTimeTrend(30);

        // Pass the data to the view
        return view('dashboard', compact(
            'totalSurat',
            'totalProjek',
            'totalAduan',
            'surat',
            'projek',
            'aduan',
            'approvedCount',
            'responseTimeData',
            'recentApplications',
            'trendData'
            // $responseTimeData,
            // [
            //     'recentApplications' => $recentApplications,
            //     'trendData' => $trendData
            // ]
        ));
    }

    // private function calculateResponseTimeData()
    // {
    //     // Calculate average response time for all applications that have progress
    //     $avgResponseTime = DB::table('surat_permohonan')
    //         ->leftJoin('progress', 'surat_permohonan.id', '=', 'progress.surat_id')
    //         ->whereNotNull('progress.created_at')
    //         ->selectRaw('AVG(DATEDIFF(progress.created_at, surat_permohonan.tarikh_terima_jkrhl)) as avg_days')
    //         ->value('avg_days');
        
    //     // Get applications with their response times
    //     $applicationsWithResponseTime = DB::table('surat_permohonan')
    //         ->leftJoin('progress', 'surat_permohonan.id', '=', 'progress.surat_id')
    //         ->selectRaw('surat_permohonan.id, 
    //                      DATEDIFF(CURDATE(), surat_permohonan.tarikh_terima_jkrhl) as response_time')
    //         ->whereNotNull('progress.created_at')
    //         ->groupBy('surat_permohonan.id')
    //         ->get();

    //     // dd($applicationsWithResponseTime);
        
    //     // Categorize response times
    //     $fastResponseCount = $applicationsWithResponseTime->where('response_time', '<=', 10)->count();
    //     $mediumResponseCount = $applicationsWithResponseTime->whereBetween('response_time', [11, 14])->count();
    //     $slowResponseCount = $applicationsWithResponseTime->where('response_time', '>', 14)->count();
        
    //     return [
    //         'avgResponseTime' => $avgResponseTime ? round($avgResponseTime, 1) : 0,
    //         'fastResponseCount' => $fastResponseCount,
    //         'mediumResponseCount' => $mediumResponseCount,
    //         'slowResponseCount' => $slowResponseCount,
    //     ];
    // }
    private function calculateResponseTimeData()
    {
        // 1. Calculate average response time (combined)
        // $avgSurat = DB::table('surat_permohonan')
        //     ->join('progress', function ($join) {
        //         $join->on('surat_permohonan.id', '=', 'progress.surat_id')
        //             ->where('progress.surat_type', '=', \App\Models\SuratPermohonan::class);
        //     })
        //     ->whereNotNull('progress.created_at')
        //     ->selectRaw('AVG(DATEDIFF(progress.created_at, surat_permohonan.tarikh_terima_jkrhl)) as avg_days');

        // $avgAduan = DB::table('aduan')
        //     ->join('progress', function ($join) {
        //         $join->on('aduan.id', '=', 'progress.surat_id')
        //             ->where('progress.surat_type', '=', \App\Models\Aduan::class);
        //     })
        //     ->whereNotNull('progress.created_at')
        //     ->selectRaw('AVG(DATEDIFF(progress.created_at, aduan.tarikh_terima_jkrhl)) as avg_days');

        // $combinedAvg = DB::table(DB::raw("({$avgSurat->toSql()} UNION ALL {$avgAduan->toSql()}) as avg_union"))
        //     ->mergeBindings($avgSurat)
        //     ->mergeBindings($avgAduan)
        //     ->selectRaw('AVG(avg_days) as avg_days')
        //     ->value('avg_days');

        // 2. Get applications with response time (combined)
        $suratApps = DB::table('surat_permohonan')
            ->join('progress', function ($join) {
                $join->on('surat_permohonan.id', '=', 'progress.surat_id')
                    ->where('progress.surat_type', '=', \App\Models\SuratPermohonan::class);
            })
            ->selectRaw("
                surat_permohonan.id,
                'SuratPermohonan' as jenis,
                DATEDIFF(progress.created_at, surat_permohonan.tarikh_terima_jkrhl) as response_time
            ")
            ->whereNotNull('progress.created_at');

        $aduanApps = DB::table('aduans')
            ->join('progress', function ($join) {
                $join->on('aduans.id', '=', 'progress.surat_id')
                    ->where('progress.surat_type', '=', \App\Models\Aduan::class);
            })
            ->selectRaw("
                aduans.id,
                'Aduan' as jenis,
                DATEDIFF(progress.created_at, aduans.tarikh_terima_jkrhl) as response_time
            ")
            ->whereNotNull('progress.created_at');

        $applicationsWithResponseTime = $suratApps
            ->unionAll($aduanApps)
            ->get();

        // 3. Categorize response times
        $fastResponseCount = $applicationsWithResponseTime->where('response_time', '<=', 10)->count();
        $mediumResponseCount = $applicationsWithResponseTime->whereBetween('response_time', [11, 14])->count();
        $slowResponseCount = $applicationsWithResponseTime->where('response_time', '>', 14)->count();

        return [
            // 'avgResponseTime' => $combinedAvg ? round($combinedAvg, 1) : 0,
            'fastResponseCount' => $fastResponseCount,
            'mediumResponseCount' => $mediumResponseCount,
            'slowResponseCount' => $slowResponseCount,
        ];
    }


    // private function getRecentApplicationsWithResponseTime($limit = 10)
    // {
    //     return DB::table('surat_permohonan')
    //         ->leftJoin('progress', function($join) {
    //             $join->on('surat_permohonan.id', '=', 'progress.surat_id')
    //                  ->whereRaw('progress.created_at = (
    //                      SELECT MIN(created_at) 
    //                      FROM progress p2 
    //                      WHERE p2.surat_id = surat_permohonan.id
    //                  )');
    //         })
    //         ->selectRaw('surat_permohonan.*, 
    //                      DATEDIFF(progress.created_at, surat_permohonan.created_at) as response_time,
    //                      DATEDIFF(NOW(), surat_permohonan.tarikh_terima_jkrhl) as tempoh_masa')
    //         ->where('surat_permohonan.status', 1)
    //         ->orderBy('surat_permohonan.created_at', 'desc')
    //         ->limit($limit)
    //         ->get();
    // }

    // private function getRecentApplicationsWithResponseTime($limit = 10)
    // {
    //     $firstProgress = DB::table('progress as p1')
    //         ->select('p1.surat_id', DB::raw('MIN(p1.created_at) as first_progress_created_at'))
    //         ->groupBy('p1.surat_id');

    //     return DB::table('surat_permohonan')
    //         ->leftJoinSub($firstProgress, 'fp', function ($join) {
    //             $join->on('surat_permohonan.id', '=', 'fp.surat_id');
    //         })
    //         ->selectRaw('surat_permohonan.*, 
    //                     DATEDIFF(fp.first_progress_created_at, surat_permohonan.tarikh_terima_bahagian) as response_time,
    //                     DATEDIFF(NOW(), surat_permohonan.tarikh_terima_bahagian) as tempoh_masa')
    //         ->where('surat_permohonan.status', 1)
    //         ->orderBy('surat_permohonan.tarikh_terima_bahagian', 'desc')
    //         ->limit($limit)
    //         ->get();
    // }
    private function getRecentApplicationsWithResponseTime($limit = 10)
    {
        $surat = SuratPermohonan::with(['latestProgress', 'assignedUser']) // adjust 'assignedUser' if relationship exists
            ->get()
            ->map(function ($item) {
                return (object)[
                    'type' => 'SuratPermohonan',
                    'id' => $item->id,
                    'no_siri_rujukan' => $item->no_siri_rujukan,
                    'tarikh_permohonan' => $item->tarikh_permohonan,
                    'status' => $item->status,
                    'assigned_user_name' => optional($item->assignedUser)->name,
                    'tempoh_masa' => $item->latestProgress 
                        ? $item->latestProgress->created_at->diffInDays($item->tarikh_terima_jkrhl)
                        : now()->diffInDays($item->tarikh_terima_jkrhl),
                    'created_at' => $item->created_at,
                ];
            });

        $aduan = Aduan::with(['latestProgress', 'assignedUser']) // adjust 'assignedUser' if exists
            ->get()
            ->map(function ($item) {
                return (object)[
                    'type' => 'Aduan',
                    'id' => $item->id,
                    'no_siri_rujukan' => $item->no_siri_rujukan,
                    'tarikh_permohonan' => $item->tarikh_permohonan,
                    'status' => $item->status,
                    'assigned_user_name' => optional($item->assignedUser)->name,
                    'tempoh_masa' => $item->latestProgress 
                        ? $item->latestProgress->created_at->diffInDays($item->tarikh_terima_jkrhl)
                        : now()->diffInDays($item->tarikh_terima_jkrhl),
                    'created_at' => $item->created_at,
                ];
            });

        return $surat->merge($aduan)
                    ->sortByDesc('created_at')
                    ->take($limit)
                    ->values(); // reset keys
    }

    private function getResponseTimeTrend($days = 30)
    {
        $startDate = Carbon::now()->subDays($days);
        
        $trendData = DB::table('surat_permohonan')
            ->leftJoin('progress', function($join) {
                $join->on('surat_permohonan.id', '=', 'progress.surat_id')
                     ->whereRaw('progress.created_at = (
                         SELECT MIN(created_at) 
                         FROM progress p2 
                         WHERE p2.surat_id = surat_permohonan.id
                     )');
            })
            ->selectRaw('DATE(surat_permohonan.created_at) as date, 
                         AVG(DATEDIFF(progress.created_at, surat_permohonan.created_at)) as avg_response_time')
            ->where('surat_permohonan.created_at', '>=', $startDate)
            ->whereNotNull('progress.created_at')
            ->groupBy('date')
            ->orderBy('date')
            ->get()
            ->map(function($item) {
                return [
                    'date' => $item->date,
                    'avg_response_time' => $item->avg_response_time ? round($item->avg_response_time, 1) : 0
                ];
            });

        // If no data, return empty array
        if ($trendData->isEmpty()) {
            return [];
        }

        return $trendData->toArray();
    }

    // Optional: Add method to get response time data via AJAX for filtering
    public function getResponseTimeTrendAjax(Request $request)
    {
        $days = $request->get('days', 30);
        $trendData = $this->getResponseTimeTrend($days);
        
        return response()->json($trendData);
    }

}
