<?php

namespace App\Http\Controllers;

use App\Models\Category;
use App\Models\Image;
use App\Models\Serial;
use App\Models\Inventory;
use App\Models\Location;
use App\Models\SubCategory;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\File;

class SerialController extends Controller
{
    public function index()
    {
        $serials = Serial::all();
        return view('serials.index', compact('serials'));
    }

    public function create($id)
    {
        $inventory = Inventory::findOrFail($id);
        $totalQuantity = $inventory->totalQuantity;

        $locations = Location::all();
        $category = Category::findOrFail($inventory->category_id);
        $subCategory = SubCategory::findOrFail($inventory->subCategory_id);

        $purchaseDate = new \DateTime($inventory->buy_date);
        $year = $purchaseDate->format('y');
        $month = $purchaseDate->format('m');

        $tagNumber = "{$category->code}-{$subCategory->code}-{$inventory->seqNo}-{$year}-{$month}-";

        $serialCount = $inventory->serials->count();
        $loopCount = $totalQuantity - $serialCount;

        // Ensure loopCount is not negative
        $loopCount = max($loopCount, 0);

        return view('serial.serialAdd', compact('inventory', 'totalQuantity', 'locations', 'tagNumber', 'loopCount'));
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'inventory_id' => 'required|exists:inventories,id',
            'serialNo' => 'required|array',
            'serialNo.*' => 'required|string',
            'tagNumber' => 'required|array',
            'tagNumber.*' => 'required|string',
            'holder' => 'required|array',
            'holder.*' => 'required|string',
            'location' => 'required|array',
            'location.*' => 'required|exists:locations,id',
            'status' => 'required|array',
            'status.*' => 'required|in:In Use/Active,Damaged,Lost/Not Found',
    
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        $validatedData = $validator->validated();
        $inventoryId = $validatedData['inventory_id'];

        if (!empty($validatedData['serialNo'])) {
            foreach ($validatedData['serialNo'] as $index => $serialNo) {
                if (isset($validatedData['holder'][$index]) && isset($validatedData['location'][$index])) {
                    Serial::create([
                        'inventory_id' => $inventoryId,
                        'serialNo' => $serialNo,
                        'holder' => $validatedData['holder'][$index],
                        'location_id' => $validatedData['location'][$index],
                        'tagNumber' => $validatedData['tagNumber'][$index],
                        'status' => $validatedData['status'][$index],
                    ]);
                }
            }
        }

        return redirect()->route('inventory.show', [$inventoryId])->with('success', 'Serial number(s) added successfully.');
    }

    public function generateTagNumber($id)
    {
        $serials = Serial::where('inventory_id', $id)->get();
        $inventories = Inventory::with('category', 'subCategories')->findOrFail($id);

        $category = Category::findOrFail($inventories->category_id);
        $subCategory = SubCategory::findOrFail($inventories->subCategory_id);

        $totalQuantity = $inventories->totalQuantity;
        $categoryCode = $category->code;
        $subCategoryCode = $subCategory->code;

        $purchaseDate = new \DateTime($inventories->buy_date);
        $year = $purchaseDate->format('y');
        $month = $purchaseDate->format('m');

        $i = 0;
        foreach ($serials as $serial) {
            $i++;
            $tagNumber = "{$categoryCode}-{$subCategoryCode}-{$inventories->seqNo}-{$year}-{$month}-{$i}/{$totalQuantity}";
            $serial->tagNumber = $tagNumber;
            $serial->save();
        }

        return redirect()->route('inventory.show', ['id1' => $id]);
    }

    public function show($id1, $id2)
    {
        $inventories = Inventory::with('category', 'subCategories')->findOrFail($id1);
        $serial = Serial::with('images')->findOrFail($id2);

        return view('serial.serialInfo', compact('inventories', 'serial', 'id1', 'id2'));
    }

    public function edit($id1, $id2)
    {
        $inventories = Inventory::with('category', 'subCategories')->findOrFail($id1);
        $serial = Serial::with('images')->findOrFail($id2);
        $locations = Location::all();

        return view('serial.editSerial', compact('serial', 'locations', 'id1', 'id2', 'inventories'));
    }

    public function update(Request $request, $id1, $id2)
{
    $serial = Serial::findOrFail($id2);
    $validatedData = $request->validate([
        'serialNo' => 'required|string',
        'tagNumber' => 'required|string',
        'holder' => 'required|string',
        'location' => 'required|exists:locations,id',
        'status' => 'required|in:In Use/Active,Damaged,Lost/Not Found',
        'photos.*' => 'image|mimes:jpeg,png,jpg,gif,svg|max:15360'
    ]);

    $photos = $request->file('photos');
    $totalSize = 0;

    if ($photos) {
        foreach ($photos as $photo) {
            $totalSize += $photo->getSize();
        }

        if ($totalSize > 15360 * 1024) {
            return redirect()->back()->withErrors(['photos' => 'The total size of all photos must not exceed 15MB.']);
        }
    }

    $serial->serialNo = $validatedData['serialNo'];
    $serial->tagNumber = $validatedData['tagNumber'];
    $serial->holder = $validatedData['holder'];
    $serial->location_id = $validatedData['location'];
    $serial->status = $validatedData['status'];
    $serial->save();

    if ($photos) {
        $inventory = Inventory::findOrFail($serial->inventory_id);

        // Find the existing images or create a new instance if none exist
        $existingImages = Image::where('serial_id', $serial->id)->first() ?? new Image(['serial_id' => $serial->id]);

        // Determine which image fields are empty and can be filled
        $imageFields = ['image1', 'image2', 'image3', 'image4', 'image5', 'image6', 'image7'];
        $availableFields = [];
        
        foreach ($imageFields as $imageField) {
            if (empty($existingImages->$imageField)) {
                $availableFields[] = $imageField;
            }
        }

        // Handle new images by filling available slots
        foreach ($availableFields as $index => $imageField) {
            if (isset($photos[$index])) {
                $photo = $photos[$index];
                $folderName = DB::connection()->getName() === 'prooffice' ? 'prooffice' : 'proview';
                $photoName = $imageField . '.' . $photo->getClientOriginalExtension();
                $path = $photo->storeAs('photos/' . $folderName . '/' . $inventory->id . '/' . $serial->id, $photoName, 'public');
                $existingImages->$imageField = $path;
            }
        }

        $existingImages->save();
    }

    return redirect()->route('serial.show', [$id1, $id2])->with('success', 'Serial updated successfully.');
}

public function deleteImage($serialId, $index)
{
    // Find the Serial and the related image by $index
    $serial = Serial::find($serialId);
    $imageField = 'image' . $index;

    if ($serial && !empty($serial->images->first()->$imageField)) {
        // Delete the image file
        Storage::delete($serial->images->first()->$imageField);
        // dd('images');
        // Remove image path from the database
        $serial->images->first()->update([$imageField => null]);

        return response()->json(['success' => true, 'message' => 'Image deleted successfully.']);
    } else {
        return response()->json(['success' => false, 'message' => 'Image not found or already deleted.']);
    }
}

    public function destroy($id)
    {
        $serial = Serial::findOrFail($id);
        $serial->delete();

        $id1 = $serial->inventory_id;
        return redirect()->route('inventory.show', [$id1])->with('success', 'Serial deleted successfully.');
    }

    
}
