<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

use App\Models\Aduan;
use App\Models\User;
use App\Models\KategoriSurat;


class AduanController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $aduan = Aduan::all();
        return view('aduan.index', compact('aduan'));
    }

    public function create()
    {
        $katSurat = KategoriSurat::all();

        return view('aduan.create', compact('katSurat'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'tajuk' => 'required|string',
            'penerangan' => 'required|string',
            'pemohon' => 'required|string|max:255',
            'syarikat' => 'required|string|max:255',
            'tarikh_permohonan' => 'required|date',
            'attachment' => 'required|file|mimes:pdf,jpg,png|max:2048',
        ]);

        // First, create the record without the attachment
        $aduan = Aduan::create(array_merge($validated, ['attachment' => null]));

        // Define a unique folder for this surat inside aduan/
        $folderName = 'aduan/surat-' . $aduan->id;

        if ($request->hasFile('attachment')) {
            $file = $request->file('attachment');
            $filename = $file->getClientOriginalName();

            // Store in subfolder
            // $path = $file->storeAs($folderName, $filename, 'public');

            $fileContents = file_get_contents($file->getPathname());
            $path = $folderName . '/' . $filename;

            Storage::disk('public')->put($path, $fileContents);

            // Update the attachment path in the record
            $aduan->update(['attachment' => $path]);
        }
    
        return redirect()->route('aduan.index')->with('Berjaya', 'Projek berjaya didaftarkan.');
    }

    /**
     * Display the specified resource.
     */
    public function show(Aduan $aduan)
    {
        $users = User::all(); // or filter by role if needed

        return view('aduan.show', compact('aduan', 'users')); // Return to show view
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Aduan $aduan)
    {
        $katSurat = KategoriSurat::all();

        return view('aduan.create', compact('aduan', 'katSurat')); // Return to edit view
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Aduan $aduan)
    {
        $validated = $request->validate([
            'tajuk' => 'required|string',
            'penerangan' => 'required|string',
            'pemohon' => 'required|string|max:255',
            'syarikat' => 'required|string|max:255',
            'tarikh_permohonan' => 'required|date',
            'attachment' => 'required|file|mimes:pdf,jpg,png|max:2048',
        ]);

        if ($request->hasFile('attachment')) {
            $file = $request->file('attachment');
            $filename = time() . '_' . $file->getClientOriginalName();

            // Define the subfolder path
            $folderName = 'aduan/surat-' . $aduan->id;

            // Ensure the directory exists (optional, Laravel usually does this automatically)
            Storage::disk('public')->makeDirectory($folderName);

            // Delete old attachment if it exists
            if ($aduan->attachment && Storage::disk('public')->exists($aduan->attachment)) {
                Storage::disk('public')->delete($aduan->attachment);
            }

            // Store the new attachment in the folder
            // $validated['attachment'] = $file->storeAs($folderName, $filename, 'public');
            $fileContents = file_get_contents($file->getPathname());
            $path = $folderName . '/' . $filename;

            Storage::disk('public')->put($path, $fileContents);

            $validated['attachment'] = $path;

        }

        $aduan->update($validated);

        return redirect()->route('aduan.index')->with('Berjaya', 'Projek berjaya dikemaskini.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Aduan $aduan)
    {
        $aduan->delete(); // Delete the record
        return redirect()->route('aduan.index')->with('Berjaya', 'Projek berjaya dipadam.');
    }

    public function storeProgress(Request $request, $id)
    {
        $validated = $request->validate([
            'progress_note' => 'required|string',
            'progress_attachment' => 'required|file|mimes:pdf,jpg,png|max:2048',
            'status' => 'nullable|integer'
        ]);

        // Get the surat permohonan record
        $aduan = Aduan::findOrFail($id);

        if ($request->hasFile('progress_attachment')) {
            $file = $request->file('progress_attachment');
            $filename = now()->format('Ymd_His') . '_' . $file->getClientOriginalName();

            // Define folder based on surat id (same logic as store method)
            $folder = 'aduan/surat-' . $aduan->id . '/progress';

            // Store the file in the specific surat folder
            // $path = $file->storeAs($folder, $filename, 'public');
            // $validated['progress_attachment'] = $path;
            $fileContents = file_get_contents($file->getPathname());
            $path = $folder . '/' . $filename;

            Storage::disk('public')->put($path, $fileContents);

            // Update the attachment path in the record
            $validated['attachment'] = $path;
        }

        // $validated['surat_permohonan_id'] = $aduan->id;
        // Progress::create($validated);
        $aduan->progresses()->create($validated); //using morph relationship - nur 8/7/2025

        // Update status in surat_permohonan table (if status is present)
        if (!is_null($validated['status'])) {
            $aduan->update(['status' => $validated['status']]);
        }

        return redirect()->route('aduan.show', $aduan->id)
            ->with('Berjaya', 'Status surat berjaya ditambah.');
    }

    public function updatePegawai(Request $request, $id)
    {
        $aduan = Aduan::findOrFail($id);

        $validated = $request->validate([
            'assign_to' => 'required|exists:users,id',
        ]);

        $aduan->update($validated);

        return redirect()->route('aduan.show', $aduan->id)->with('Berjaya', 'Pegawai bertugas berjaya dikemaskini.');
    }

}
