<?php

namespace App\Http\Controllers;

use App\Models\PermohonanProjek; // Import the model
use Illuminate\Http\Request;
use App\Models\Milestone; // Import the Milestone model
use Illuminate\Support\Facades\Storage; // Import Storage facade

class PermohonanProjekController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $model = PermohonanProjek::all(); // Fetch all records
        return view('permohonan-projek.index', compact('model')); // Return to index view
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('permohonan-projek.create'); // Return to create view
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        // dd((new PermohonanProjek())->getTable());

        $validated = $request->validate([
            'tajuk' => 'required|string|max:255',
            'penerangan' => 'required|string',
            'pemohon' => 'required|string|max:255',
            'syarikat' => 'required|string|max:255',
            'tarikh_permohonan' => 'required|date',
            'attachment' => 'nullable|file|mimes:pdf,jpg,png|max:2048',
        ]);

        // Handle file upload if exists
        if ($request->hasFile('attachment')) {
            $file = $request->file('attachment');
            $filename = $file->getClientOriginalName(); // Get the original file name
            $validated['attachment'] = $file->storeAs('permohonan-projek', $filename, 'public');
        }

        PermohonanProjek::create($validated); // Save the record
        return redirect()->route('permohonanProjek.index')->with('success', 'Projek berjaya didaftarkan.');
    }

    /**
     * Display the specified resource.
     */
    public function show(PermohonanProjek $permohonanProjek)
    {
        return view('permohonan-projek.show', compact('permohonanProjek')); // Return to show view
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(PermohonanProjek $permohonanProjek)
    {
        return view('permohonan-projek.create', compact('permohonanProjek')); // Return to edit view
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, PermohonanProjek $permohonanProjek)
    {
        $validated = $request->validate([
            'tajuk' => 'required|string|max:255',
            'penerangan' => 'required|string',
            'pemohon' => 'required|string|max:255',
            'syarikat' => 'required|string|max:255',
            'tarikh_permohonan' => 'required|date',
            'attachment' => 'nullable|file|mimes:pdf,jpg,png|max:2048',
        ]);

        // Handle file upload if exists
        if ($request->hasFile('attachment')) {
            $file = $request->file('attachment');
            $filename = time() . '_' . $file->getClientOriginalName(); // Add timestamp to avoid overwrites
    
            // Optionally delete the old file if it exists
            if ($permohonanProjek->attachment && \Storage::disk('public')->exists($permohonanProjek->attachment)) {
                \Storage::disk('public')->delete($permohonanProjek->attachment);
            }
    
            // Store the file with the original filename
            $validated['attachment'] = $file->storeAs('permohonan-projek', $filename, 'public');
        }

        $permohonanProjek->update($validated); // Update the record
        return redirect()->route('permohonanProjek.index')->with('success', 'Projek berjaya dikemaskini.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(PermohonanProjek $permohonanProjek)
    {
        $permohonanProjek->delete(); // Delete the record
        return redirect()->route('permohonanProjek.index')->with('success', 'Projek berjaya dipadam.');
    }
    
    public function showMilestone($id)
    {
        $milestones = Milestone::where('project_id', $id)->get();
        return view('permohonan-projek.milestone', compact('milestones', 'id'));
    }

    // public function storeMilestone(Request $request, $id)
    // {        
    //     Milestone::create([
    //         'project_id' => $id,
    //         'phase' => $request->phase,
    //         'penerangan' => $request->penerangan,
    //         'tarikh' => $request->tarikh,
    //     ]);
    //     return back()->with('success', 'Milestone added successfully.');
    // }

    // public function storeMilestone(Request $request, $id)
    // {
    //     if ($request->has('new')) {
    //         $phases = $request->input('new.phase');
    //         $penerangans = $request->input('new.penerangan');
    //         $tarikhs = $request->input('new.tarikh');

    //         for ($i = 0; $i < count($phases); $i++) {
    //             // Skip empty rows
    //             if (empty($phases[$i]) && empty($penerangans[$i]) && empty($tarikhs[$i])) {
    //                 continue;
    //             }

    //             Milestone::create([
    //                 'project_id' => $id,
    //                 'phase' => $phases[$i],
    //                 'penerangan' => $penerangans[$i],
    //                 'tarikh' => $tarikhs[$i],
    //             ]);
    //         }
    //     }

    //     return back()->with('success', 'Milestone(s) added successfully.');
    // }
    public function storeMilestone(Request $request, $id)
    {
        // 1. Handle New Milestones
        if ($request->has('new')) {
            $phases = $request->input('new.phase');
            $penerangans = $request->input('new.penerangan');
            $tarikhs = $request->input('new.tarikh');

            for ($i = 0; $i < count($phases); $i++) {
                // Skip empty rows
                if (empty($phases[$i]) && empty($penerangans[$i]) && empty($tarikhs[$i])) {
                    continue;
                }

                Milestone::create([
                    'project_id' => $id,
                    'phase' => $phases[$i],
                    'penerangan' => $penerangans[$i],
                    'tarikh' => $tarikhs[$i],
                ]);
            }
        }

        // 2. Handle Existing Milestones Update
        if ($request->has('existing')) {
            foreach ($request->input('existing') as $milestoneId => $data) {
                $milestone = Milestone::find($milestoneId);

                if ($milestone) {
                    $milestone->update([
                        'phase' => $data['phase'] ?? null,
                        'penerangan' => $data['penerangan'] ?? null,
                        'tarikh' => $data['tarikh'] ?? null,
                    ]);
                }
            }
        }

        return back()->with('success', 'Milestone(s) saved successfully.');
    }

    public function updateMilestone(Request $request, $id)
    {
        $milestone = Milestone::findOrFail($request->milestone_id);
        $milestone->update($request->only(['phase', 'penerangan', 'tarikh']));
        return back()->with('success', 'Milestone updated successfully.');
    }

    // public function destroyMilestone($id, Request $request)
    // {
    //     Milestone::findOrFail($request->milestone_id)->delete();
    //     return back()->with('success', 'Milestone deleted successfully.');
    // }
    public function destroyMilestone(Request $request, Milestone $milestone)
    {
        $milestone->delete();
        return back()->with('success', 'Milestone deleted successfully.');
    }

}